# Copyright (C) 2004-2008 Daniel Vrit

# This file is part of Manitou-Mail (see http://www.manitou-mail.org)

# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License version 2 as
# published by the Free Software Foundation.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330,
# Boston, MA 02111-1307, USA.

package Manitou::Attachments;

use strict;
use vars qw(@ISA @EXPORT_OK);

use File::stat;
use Carp;
use POSIX qw(tmpnam);
use Manitou::Encoding qw(encode_dbtxt header_decode);
use Digest::SHA1;

require Exporter;
@ISA = qw(Exporter);
@EXPORT_OK = qw(flatten_and_insert_attach
		detach_text_attachments
		attach_parts);

sub get_sequence_nextval {
  my ($dbh, $seq) = @_;
  my ($nextval, $sth, $row);
  $sth=$dbh->prepare("SELECT nextval('".$seq."')") or die $dbh->errstr;
  $sth->execute() or die $dbh->errstr;
  my @row = $sth->fetchrow_array;
  if ($row[0]) {
    $nextval = $row[0];
  } else {
    $nextval = 1;
  }
  $sth->finish;
  return $nextval;
}

sub detach_text_attachments {
  my ($dbh, $top, $mail_id, $pbody_text) = @_;
  my $attachments;
  if ($top->is_multipart) {
    foreach my $p ($top->parts) {
      if ($p->effective_type eq "text/plain" &&
	  $p->head->recommended_filename eq "" &&
	  $p->bodyhandle) {
	my $charset = $p->head->mime_attr("content-type.charset") || 'iso-8859-1';
	$charset='iso-8859-1' if (!Encode::resolve_alias($charset));
	$$pbody_text .= Encode::decode($charset, $p->bodyhandle->as_string);
      }
      else {
	$attachments += detach_text_attachments($dbh, $p, $mail_id, $pbody_text);
      }
    }
  }
  else {
    $attachments = flatten_and_insert_attach($dbh, $top, $mail_id);
  }
  return $attachments;
}

sub flatten_and_insert_attach {
  my ($dbh, $top,$mail_id) = @_;
  my $attachments;
  if ($top->is_multipart) {
    foreach ($top->parts) {
      $attachments+=flatten_and_insert_attach($dbh, $_, $mail_id);
    }
  }
  else {
    if ($top->bodyhandle) {
      &insert_attachment($dbh, $mail_id, $top);
      $attachments++;
    }
  }
  return $attachments;
}

# remove undesirable characters
sub sanitize_filename {
  my $f=$_[0];
  $f =~ s/\/\\\"\'\<\>//g;
  $f =~ s/\s/_/g;
  $f =~ tr/\x00-\x1F//d;
  return $f;
}

sub insert_attachment {
  my ($dbh, $mail_id, $mime_obj) = @_;
  my $attachment_id = get_sequence_nextval($dbh, "seq_attachment_id");
  my $lobjId;
  my $attch_file=tmpnam();
  open(PGIN, ">$attch_file") or die "can not open $attch_file: $!";
  $mime_obj->bodyhandle->print(\*PGIN);
  close(PGIN);
  my $filesize = stat($attch_file)->size;

  my $stha = $dbh->prepare("INSERT INTO attachments(attachment_id,mail_id,content_type,content_size,filename,charset,mime_content_id) VALUES (?,?,?,?,?,?,?)")  or die $dbh->errstr;

  my $pos = 0;
  $stha->bind_param(++$pos, $attachment_id);
  $stha->bind_param(++$pos, $mail_id);
  my $a_type=substr(header_decode($mime_obj->effective_type),0,300);
  $stha->bind_param(++$pos, encode_dbtxt($a_type));
  $stha->bind_param(++$pos, $filesize);

  my $fname=header_decode($mime_obj->head->recommended_filename);
  $fname=sanitize_filename(substr($fname,0,300));
  $stha->bind_param(++$pos, encode_dbtxt($fname));

  my $charset=header_decode($mime_obj->head->mime_attr("content-type.charset"));
  $stha->bind_param(++$pos, encode_dbtxt(substr($charset,0,30)));
  LogError($stha->errstr) if $stha->err;

  my $content_id=$mime_obj->get("Content-ID");
  # content-ID syntax must be <addr-spec> (RFC2111)
  $content_id = ($content_id =~ /^\<(.*)\>$/) ? $1 : undef;
  $stha->bind_param(++$pos, encode_dbtxt(header_decode($content_id)));

  $stha->execute or die $stha->errstr;
  $stha->finish;

  if ($filesize>0) {
    # compute the fingerprint
    my $sha1 = Digest::SHA1->new;
    open(PGIN, "$attch_file") or die "can not open $attch_file: $!";
    binmode PGIN;
    $sha1->addfile(*PGIN);
    my $fingerprint = $sha1->b64digest;
    # check if the content already exists in the database
    my $sth1=$dbh->prepare("SELECT content FROM attachment_contents WHERE fingerprint=?");
    $sth1->execute($fingerprint) or die $sth1->errstr;
    ($lobjId)=$sth1->fetchrow_array;
    $sth1->finish;
    if (!$lobjId) {
      # import the contents
      $lobjId = $dbh->func($attch_file, 'lo_import');
    }

    my $sth = $dbh->prepare("INSERT INTO attachment_contents(attachment_id, content, fingerprint) VALUES (?,?,?)") or die $dbh->errstr;
    $sth->bind_param(1, $attachment_id);
    $sth->bind_param(2, $lobjId);
    $sth->bind_param(3, $fingerprint);

    $sth->execute or die $sth->errstr;
    $sth->finish;
  }
  unlink($attch_file);
}

# Create the file if it is a file or put the attachment contents to
# the location pointed to by $content
sub get_one_attachment {
  my ($dbh, $a_id,$filename,$content_size,$content,$tmpdir)=@_;
  my $ret;
  my $sth = $dbh->prepare ("SELECT content FROM attachment_contents WHERE " .
			   "attachment_id=?") ||
			     die "Can't prepare statement: $DBI::errstr";
  $sth->execute($a_id) || die "Can't execute statement: $DBI::errstr";

  my @row = $sth->fetchrow_array;
  die $sth->errstr if $sth->err;
  my $lobjId = $row[0];
  $sth->finish;

  # Note: this needs to run while inside a transaction (this is required
  # for the lo_* functions)
  if ($filename) {
    $ret = $dbh->func ($lobjId, $tmpdir . "/$filename", 'lo_export');
  }
  else {
    $$content = "";
    my $lobj_fd = $dbh->func ($lobjId, $dbh->{pg_INV_READ}, 'lo_open');
    die $dbh->errstr if (!defined($lobj_fd));
    my $buf;
    my $nbytes;
    while ($content_size > 0) {
      $nbytes = $dbh->func($lobj_fd, $buf, 16384, 'lo_read');
      die $dbh->errstr if (!defined($nbytes));
      $content_size -= $nbytes;
      $$content .= $buf;
    }
    $dbh->func ($lobj_fd, 'lo_close');
  }
}

sub attach_parts {
  my ($dbh,$mail_id,$mobj,$tmpdir)=@_;

  my $sth = $dbh->prepare ("SELECT attachment_id,content_type,content_size,filename FROM attachments WHERE mail_id=?") || die "Can't prepare statement: $DBI::errstr";
  $sth->execute($mail_id) || die "Can't execute statement: $DBI::errstr";
  while (my @row = $sth->fetchrow_array) {
    my $attch_data;
    my $filename=$row[3];
    get_one_attachment($dbh, $row[0], $filename, $row[2], \$attch_data,$tmpdir);
    if ($filename) {
      $mobj->attach(Path    => $tmpdir . "/" . $filename,
		   Encoding    => '-SUGGEST',
		   Type => $row[1]);
    } else {
      $mobj->attach(Data    => $attch_data,
		   Encoding    => '-SUGGEST',
		   Type => $row[1]);
    }
  }
  $sth->finish;
}
